#!/bin/bash
#set -x

################################################################################
# Script to automatically download and install the Mac agent using the values
# set in the set_global_variables function
################################################################################

set_global_variables()
{
    # Registration token for the Client / Site the Mac should be placed under
    param_registration_token=""

    # IP / FQDN of the server without any protocol prefix or trailing slash
    param_server=""
    param_protocol=""
    param_port=

    # Set to true if using NC release 2024.3 onwards, this will download the latest Mac Agent from the provided server
    param_download_flag=false

    # A download url for the Mac agent, at the time of writing, 1.11.0.562 is the latest Mac Agent
    param_download_url="${param_protocol}://${param_server}/download/1.11.0.562/macosx/N-central/Install_N-central_Agent_v1.11.0.562.dmg"

    using_urls_list=false
    
    # ID of the Customer / Site that the Agent should be installed under. 
    # Get this from the 'Access Code' column in the Administration -> Customers screen at Service Organisation level in N-central
    PARAM_CUSTOMER_ID=""
    
    # The name of the Customer / Site that the Agent should be installed under.
    # Get this from the 'Name' column in the Administration -> Customers screen at Service Organisation level in N-central
    PARAM_CUSTOMER_NAME=""
}

set_constants() 
{
    readonly LOG_DIR="/Library/Logs/N-central Agent"
    readonly AGENT_DIR="/Library/N-central Agent"
    readonly LEGACY_AGENT_DIR="/Applications/Mac_Agent.app"

    readonly REGISTER_WITH_SERVER_PARAMS_FILEPATH="/tmp/ncentral_silent_install.params"
    readonly DMG_MOUNT_POINT="/Volumes/N-central Agent silent installation"
    readonly IMAGE_BASE_VOLUME_NAME="Install N-central Agent"

    readonly INSTALLER_PACKAGE_FILENAME="Install.pkg"

    readonly LOG_FILENAME="nagent.silent_install.log"
    readonly LOG_FILEPATH="${LOG_DIR}/${LOG_FILENAME}"

    readonly SILENT_INSTALL_OUTPUT_FILENAME="nagent.silent_install-output.log"
    readonly SILENT_INSTALL_OUTPUT_FILEPATH="${LOG_DIR}/${SILENT_INSTALL_OUTPUT_FILENAME}"

    THIS_SCRIPT_FILENAME=$(basename "$0")
    readonly THIS_SCRIPT_FILENAME

    # Return error codes

    readonly EXIT_ERROR_INVALID_PARAMETERS=1
    readonly EXIT_ERROR_NO_INSTALLER=2
    readonly EXIT_ERROR_NO_SUDO=3
    readonly EXIT_ERROR_AGENT_ALREADY_INSTALLED=5
}


################################################################################
# Functions
################################################################################

download_installer() {
    if [ "${param_download_flag}" = true ]; then
        download_url="${param_protocol}://${param_server}/download/latest/macosx/N-central/Install_N-central_Agent.dmg"
        echo "Downloading installer from ${download_url}"
        curl -L -o /tmp/Install_N-central_Agent.dmg "${download_url}" || {
            echo "Failed to download installer from ${download_url}"
            exit ${EXIT_ERROR_NO_INSTALLER}
        }
        PARAM_INSTALLER="/tmp/Install_N-central_Agent.dmg"
    elif [ -n "${param_download_url}" ]; then
        echo "Downloading installer from ${param_download_url}"
        curl -L -o /tmp/Install_N-central_Agent.dmg "${param_download_url}" || {
            echo "Failed to download installer from ${param_download_url}, attempting to download from SIS Server"
            curl -L -o /tmp/Install_N-central_Agent.dmg "https://sis.n-able.com/GenericFiles/NcentralMacAgent/1.11.0.0/Install_N-central_Agent_v1.11.0.562.dmg" || {
                echo "Failed to download installer from SIS, exiting script"
                exit ${EXIT_ERROR_NO_INSTALLER}
            }
        }
        PARAM_INSTALLER="/tmp/Install_N-central_Agent.dmg"
    fi
}

check_if_running_as_sudo() 
{
    if [ "$(id -u)" -ne 0 ]; then
        echo "Please run as root:"
        echo "    $0"

        exit ${EXIT_ERROR_NO_SUDO}
    fi
}

setup_debug() # shell_arguments
{
    # Exit immediately if a command exits with a non-zero status.
    set -e

    if [ -n "${RMM_DEBUG}" ]; then
        echo "Running: $1"
        # Print commands and their arguments as they are executed.
        set -x
    fi
}

################################################################################

log_info() # message
{
    log_message "$1" false
}

log_warning() # message
{
    log_message "$1" true
    echo "$1"
}

log_message() # message, is_warning
(
    if [ -d "${LOG_DIR}" ]; then
        prefix="I"

        if ${2}; then
            prefix="W"
        fi

        CURRENT_TIME=$(date '+%Y-%m-%d %T')
        readonly CURRENT_TIME

        echo "[${CURRENT_TIME}] [${prefix}] $1" >> "${LOG_FILEPATH}"
    fi
)

################################################################################

decode_activation_key()
{
    DECODED_ACTIVATION_KEY=$( echo "$1" | openssl enc -base64 -d -A )
    readonly DECODED_ACTIVATION_KEY

    # Decoded key format:  https://warsteiner.lab2.n-able.com:443|37683|1|0
    URIS=$( printf "%s" "${DECODED_ACTIVATION_KEY}" | awk -F "|" '{print $1}' )
    readonly URIS
    FIRST_SERVER=$( printf "%s" "${URIS}" | awk -F "," '{print $1}')
    readonly FIRST_SERVER
    
    param_appliance=$( printf "%s" "${DECODED_ACTIVATION_KEY}" | awk -F "|" '{print $2}' )
    param_registration_token=$( printf "%s" "${DECODED_ACTIVATION_KEY}" | awk -F "|" '{print $4}' )
    param_server=$( printf "%s" "${DECODED_ACTIVATION_KEY}" | awk -F "|" '{print $1}' )
    param_protocol=$( printf "%s" "${FIRST_SERVER}" | awk -F ":" '{print $1}' )
    param_port=$( printf "%s" "${FIRST_SERVER}" | awk -F ":" '{print $3}' )

    using_urls_list=true
}

unmount_installer_dmg() 
{
    # Mounting a DMG Installer by double clicking on it resolve in a different 
    # place than expected. Trying to mount the same DMG installer under another 
    # path will give an error that the resource is busy (in use). To resolve 
    # this we need to unmount any old installers currently in use in order
    # proceed with the installation.

    # Unmount `hdiutil` mounted, silent installer volume.
    if [ -d "${DMG_MOUNT_POINT}" ]; then
        log_info "Unmounting: ${DMG_MOUNT_POINT}"

        hdiutil detach "${DMG_MOUNT_POINT}" -force -quiet
    fi

    # Unmount manually mounted volumes. https://www.shellcheck.net/wiki/SC2044
    for volume in "/Volumes"/*; do
        case ${volume} in
            *"${IMAGE_BASE_VOLUME_NAME}"*)
                log_info "Unmounting: ${volume}"

                hdiutil detach "${volume}" -force -quiet
                ;;
            *)
                ;;
        esac
    done
}

################################################################################
# Main entry point
################################################################################

main()
{
    setup_debug "$@"
    set_constants
    set_global_variables
    check_if_running_as_sudo

    mkdir -p "${LOG_DIR}"

    # Default values for port and protocol
    if [ -z "${param_port}" ]; then
        param_port=443
    fi

    if [ -z "${param_protocol}" ]; then
        param_protocol=https
    fi

    if [ -f "${AGENT_DIR}/nagent" ] || [ -d "${LEGACY_AGENT_DIR}" ]; then
        log_warning "Agent is already installed on this device. Please uninstall it first in order to use $0"

        exit ${EXIT_ERROR_AGENT_ALREADY_INSTALLED}
    fi

    if [ "${param_download_flag}" = true ] || [ -n "${param_download_url}" ]; then
        download_installer
    elif [ -n "${param_installer}" ]; then
        PARAM_INSTALLER="${param_installer}"
    else
        echo "No installer specified."
        exit ${EXIT_ERROR_NO_INSTALLER}
    fi

    if ! [ -f "${PARAM_INSTALLER}" ]; then
        log_warning "Installer doesn't exist: ${PARAM_INSTALLER}"
        print_help

        exit ${EXIT_ERROR_INVALID_PARAMETERS}
    fi

    INSTALLER_EXTENSION="${PARAM_INSTALLER##*.}"
    readonly INSTALLER_EXTENSION

    if [ "${INSTALLER_EXTENSION}" != "pkg" ] && [ "${INSTALLER_EXTENSION}" != "dmg" ]; then
        log_warning "Bad installer file. Should have .pkg or .dmg extension."
        print_help

        exit ${EXIT_ERROR_INVALID_PARAMETERS}
    fi

    if [ -z "${param_server}" ] \
       && [ -z "${PARAM_CUSTOMER_NAME}" ] \
       && [ -z "${PARAM_CUSTOMER_ID}" ] \
       && [ -z "${param_registration_token}" ];
    then
        if [ -n "${PARAM_ACTIVATION_KEY}" ]; then
            decode_activation_key "${PARAM_ACTIVATION_KEY}"
        else
            print_help

            exit ${EXIT_ERROR_INVALID_PARAMETERS}
        fi
    fi

    if 
        { \
            [ -z "${param_server}" ] \
            || [ -z "${param_port}" ] \
            || [ -z "${param_protocol}" ] \
            || [ -z "${param_registration_token}" ]; \
        } \
        || [ -z "${param_appliance}" ] \
        && { \
            [ -z "${PARAM_CUSTOMER_ID}" ] \
            || [ -z "${PARAM_CUSTOMER_NAME}" ]; \
        } 
    then
        log_warning "Not valid activation key"

        print_help

        exit ${EXIT_ERROR_INVALID_PARAMETERS}
    fi

    rm -f "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"
    # trap 'rm -f "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"' EXIT

    cat <<EOF >> "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"
NC_IVPRM_TOKEN="${param_registration_token}"
NC_IVPRM_PROXY="${PARAM_PROXY}"
EOF

    if $using_urls_list; then
        cat <<EOF >> "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"
NC_IVPRM_SERVERS_URLS="${param_server}"
EOF
    else
        cat <<EOF >> "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"
NC_IVPRM_SERVER="${param_server}"
NC_IVPRM_PORT="${param_port}"
NC_IVPRM_PROTOCOL="${param_protocol}"
EOF
    fi

    if [ -z "${param_appliance}" ]; then
        cat <<EOF >> "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"
NC_IVPRM_ID="${PARAM_CUSTOMER_ID}"
NC_IVPRM_NAME="${PARAM_CUSTOMER_NAME}"
EOF
    else
        cat <<EOF >> "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"
NC_IVPRM_APPLIANCE="${param_appliance}"
EOF
    fi

    log_info "Installer configured at '${REGISTER_WITH_SERVER_PARAMS_FILEPATH}'"
    echo "Configured!"

    # Output log cleanup
    if [ -f "${SILENT_INSTALL_OUTPUT_FILEPATH}" ]; then
        rm "${SILENT_INSTALL_OUTPUT_FILEPATH}"
    fi

    if [ "${INSTALLER_EXTENSION}" = "dmg" ]; then
        unmount_installer_dmg
        
        hdiutil attach "${PARAM_INSTALLER}" -noautoopen -nobrowse -readonly -mountpoint "${DMG_MOUNT_POINT}" -quiet

        if ! [ -d "${DMG_MOUNT_POINT}" ]; then
            log_warning "Failed to mount '${PARAM_INSTALLER}' or unsupported installer. Please use another file"

            exit ${EXIT_ERROR_NO_INSTALLER}
        fi

        trap 'rm -f "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}" && hdiutil detach "${DMG_MOUNT_POINT}" -force -quiet' EXIT

        TMP_DIR="$(mktemp -d)"
        readonly TMP_DIR
        readonly TARGET_INSTALL_FILEPATH="${TMP_DIR}/${INSTALLER_PACKAGE_FILENAME}"

        cp "${DMG_MOUNT_POINT}/${INSTALLER_PACKAGE_FILENAME}" "${TARGET_INSTALL_FILEPATH}"

        hdiutil detach "${DMG_MOUNT_POINT}" -force -quiet

        trap 'rm -f "${REGISTER_WITH_SERVER_PARAMS_FILEPATH}"' EXIT

        echo "Unpacked"

        installer -pkg "${TARGET_INSTALL_FILEPATH}" -target /
    else
        installer -pkg "${PARAM_INSTALLER}" -target /
    fi

    if [ -f "${SILENT_INSTALL_OUTPUT_FILEPATH}" ]; then
        cat "${SILENT_INSTALL_OUTPUT_FILEPATH}"
        rm "${SILENT_INSTALL_OUTPUT_FILEPATH}"
    fi
}

################################################################################

main "$@"
